<?php

namespace App\Console\Commands;

use App\Models\ExternalWmPrescriptionModel;
use App\Services\ExternalService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class WmPrescription extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'wm-prescription:init {timestamp_str?}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = '同步西药处方信息';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('处方信息（西药）同步开始...');
        $timestamp_str = $this->argument('timestamp_str') ?? '';
        // 如果 timestamp_str 为空，则获取当月的起始时间戳（精确到毫秒）
        if (empty($timestamp_str)) {
            // 获取当前月份的第一天
            $startOfMonth = Carbon::now()->startOfMonth();
            // 转换为时间戳（秒级），然后乘以 1000 转换为毫秒
            $timestamp_str = $startOfMonth->timestamp * 1000;
        }
        // 计算结束时间：基于 timestamp_str 加上 30 天
        $beginTime = Carbon::createFromTimestampMs($timestamp_str); // 将毫秒时间戳转换为 Carbon 实例
        $endTime = $beginTime->copy()->addDays(31)->timestamp * 1000; // 加上 30 天并转换为毫秒时间戳
        // 输出时间戳以供调试
        $this->info("使用的开始时间戳: {$timestamp_str}");
        $this->info("使用的结束时间戳: {$endTime}");

        try {
            $pageNum = 1;  // 初始页码
            $maxPage = 1;  // 默认最大页数为 1
            $baseParams = [
                // 根据文档填写必要的参数
                "beginTime"=>$timestamp_str,
                "endTime"=>$endTime,
                "pageNum"=>1,
                "pageSize"=>20,
            ];
            $eHospitalApi = new ExternalService(env('580_APP_KEY'), env('580_APP_SECRET'), env('580_DOMAIN'));
            while ($pageNum <= $maxPage) {
                // 设置当前页码参数
                $params = array_merge($baseParams, [
                    "pageNum" => $pageNum,
                ]);
                // 处方信息（西药）
                $response = $eHospitalApi->createPrescriptionWm($params);
                // 检查接口是否调用成功
                if (isset($response['err']) && $response['err'] === 0) {
                    echo "第 {$pageNum} 页查询成功。\n";

                    // 当前页数据
                    $response_data = $response['data']['list'];
                    Log::info('WmPrescription-处方信息（西药）:'.json_encode($response_data));
                    if( empty($response_data) ){
                        break;
                    }
                    # 处理数据存入到数据库，如果存在就更新否则就新增
                    foreach ($response_data as $item) {
                        $prescription = ExternalWmPrescriptionModel::where('hosRxCode', $item['hosRxCode'])->first();
                        if($prescription){
                            $this->info('更新处方（西药）：'. $item['hosRxCode']);
                            // 基本信息
                            $prescription->idcardTypeCode = $item['idcardTypeCode'] ?? null; // 患者就诊卡证类型
                            $prescription->idcardNo = $item['idcardNo'] ?? null; // 患者就诊卡证号码
                            $prescription->name = $item['name'] ?? null; // 姓名
                            $prescription->genderCode = $item['genderCode'] ?? null; // 性别代码
                            $prescription->birthdate = $item['birthdate'] ?? null; // 出生日期
                            $prescription->visitNo =  $item['visitNo'] ?? null; // 门诊编号
                            // 处方基本信息
                            $prescription->rxTypeCode = $item['rxTypeCode'] ?? null; // 处方分类代码
                            $prescription->rxAmount = $item['rxAmount'] ?? null; // 处方药品总金额
                            $prescription->rxDescription = $item['rxDescription'] ?? null; // 处方备注说明

                            // 开处方医疗机构信息
                            $prescription->unifiedOrgCode = $item['unifiedOrgCode'] ?? null; // 开处方医疗机构编码
                            $prescription->orgName = $item['orgName'] ?? null; // 开处方医疗机构名称

                            // 开处方科室信息
                            $prescription->deptCode = $item['deptCode'] ?? null; // 开处方科室编码
                            $prescription->deptName = $item['deptName'] ?? null; // 开处方科室名称
                            $prescription->deptClassCode = $item['deptClassCode'] ?? null; // 开处方科室对应诊疗科目编码
                            $prescription->deptClassName = $item['deptClassName'] ?? null; // 开处方科室对应诊疗科目名称

                            // 开处方医师信息
                            $prescription->doctIdcardNo = $item['doctIdcardNo'] ?? null; // 开处方医师身份证号
                            $prescription->doctName = $item['doctName'] ?? null; // 开处方医师姓名
                            $prescription->rxDatetime = $item['rxDatetime'] ?? null; // 开处方日期时间
                            $prescription->doctCaSign = $item['doctCaSign'] ?? null; // 开处方医师电子签名

                            // 数字签名相关
                            $prescription->originalData = $item['originalData'] ?? null; // 数据原文
                            $prescription->certificate = $item['certificate'] ?? null; // 数字证书

                            // 处方流向类型
                            $prescription->rotateTypeCode = $item['rotateTypeCode'] ?? null; // 处方流向类型

                            // 处方明细（对象数组）
                            if (isset($item['items'])) {
                                $prescription->items = json_encode($item['items'], JSON_UNESCAPED_UNICODE); // 处方明细数组
                            }

                            // 处方附加服务费用明细（对象数组）
                            if (isset($item['addAmountItems'])) {
                                $prescription->addAmountItems = json_encode($item['addAmountItems'], JSON_UNESCAPED_UNICODE); // 处方附加服务费用明细
                            }
                            // 是否医保处方
                            $prescription->medicalInsurance = $item['medicalInsurance'] ?? null;

                            $prescription->updated_at = Carbon::now();
                            // 保存数据
                            $prescription->save();
                        }else{
                            $this->info('新增处方（西药）：'. $item['hosRxCode']);
                            $prescription = new ExternalWmPrescriptionModel();
                            // 基本信息
                            $prescription->idcardTypeCode = $item['idcardTypeCode'] ?? null; // 患者就诊卡证类型
                            $prescription->idcardNo = $item['idcardNo'] ?? null; // 患者就诊卡证号码
                            $prescription->name = $item['name'] ?? null; // 姓名
                            $prescription->genderCode = $item['genderCode'] ?? null; // 性别代码
                            $prescription->birthdate = $item['birthdate'] ?? null; // 出生日期
                            $prescription->visitNo =  $item['visitNo'] ?? null; // 门诊编号

                            // 处方基本信息
                            $prescription->hosRxCode = $item['hosRxCode'] ?? null; // 处方编码
                            $prescription->rxTypeCode = $item['rxTypeCode'] ?? null; // 处方分类代码
                            $prescription->rxAmount = $item['rxAmount'] ?? null; // 处方药品总金额
                            $prescription->rxDescription = $item['rxDescription'] ?? null; // 处方备注说明

                            // 开处方医疗机构信息
                            $prescription->unifiedOrgCode = $item['unifiedOrgCode'] ?? null; // 开处方医疗机构编码
                            $prescription->orgName = $item['orgName'] ?? null; // 开处方医疗机构名称

                            // 开处方科室信息
                            $prescription->deptCode = $item['deptCode'] ?? null; // 开处方科室编码
                            $prescription->deptName = $item['deptName'] ?? null; // 开处方科室名称
                            $prescription->deptClassCode = $item['deptClassCode'] ?? null; // 开处方科室对应诊疗科目编码
                            $prescription->deptClassName = $item['deptClassName'] ?? null; // 开处方科室对应诊疗科目名称

                            // 开处方医师信息
                            $prescription->doctIdcardNo = $item['doctIdcardNo'] ?? null; // 开处方医师身份证号
                            $prescription->doctName = $item['doctName'] ?? null; // 开处方医师姓名
                            $prescription->rxDatetime = $item['rxDatetime'] ?? null; // 开处方日期时间
                            $prescription->doctCaSign = $item['doctCaSign'] ?? null; // 开处方医师电子签名

                            // 数字签名相关
                            $prescription->originalData = $item['originalData'] ?? null; // 数据原文
                            $prescription->certificate = $item['certificate'] ?? null; // 数字证书

                            // 处方流向类型
                            $prescription->rotateTypeCode = $item['rotateTypeCode'] ?? null; // 处方流向类型

                            // 处方明细（对象数组）
                            if (isset($item['items'])) {
                                $prescription->items = json_encode($item['items'], JSON_UNESCAPED_UNICODE); // 处方明细数组
                            }

                            // 处方附加服务费用明细（对象数组）
                            if (isset($item['addAmountItems'])) {
                                $prescription->addAmountItems = json_encode($item['addAmountItems'], JSON_UNESCAPED_UNICODE); // 处方附加服务费用明细
                            }
                            // 是否医保处方
                            $prescription->medicalInsurance = $item['medicalInsurance'] ?? null;
                            // 保存数据
                            $prescription->save();
                        }
                    }

                    print_r("maxPage:" . json_encode($response['data']['maxPage']) . "页\n");
                    // 更新分页信息
                    if (isset($response['data']['maxPage'])) {
                        $maxPage = $response['data']['maxPage'];
                    }
                    $pageNum++;
                    // if (isset($response['data']['pageNum'])) {
                    //     $pageNum = $response['data']['pageNum'] + 1; // 下一页
                    // } else {
                    //     $pageNum++; // 如果没有明确的 pageNum，手动递增
                    // }
                    print_r("pageNum:" . $pageNum . "页\n");
                }else{
                    Log::info('WmPrescription-处方信息（西药）- 接口异常' .json_encode($response));
                    echo "第 {$pageNum} 页查询失败。\n";
                    break; // 如果某页失败，停止翻页
                }

                if( $maxPage == 1 ){
                    break;
                }

            }
        }catch (\Exception $e) {
            Log::info('WmPrescription-处方信息（西药）- 代码异常' .json_encode($e));
            $this->error($e->getMessage());
        }
        $this->info('处方信息（西药）同步结束...');
    }
}
