<?php

namespace App\Admin\Controllers;

use App\Admin\Repositories\DoctorRepository;
use App\Models\DoctorModel;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Http\Controllers\AdminController;
use Dcat\Admin\Show;
use Jxlwqq\IdValidator\IdValidator;

// 医师
class DoctorController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(new DoctorRepository('user'), function (Grid $grid) {
            $grid->model()->orderBy('id', 'DESC');

            $grid->column('id')->sortable();
            $grid->column('name');
            $grid->column('status')->switch('', true);
            $grid->column('is_auto')->using(DoctorModel::IS_AUTO_MAP)->dot(DoctorModel::IS_AUTO_MAP_COLOR);
            $grid->column('id_card')->display(function ($content) {
                return data_masking($content, 'idcard');
            });
            $grid->column('mobile')->display(function ($content) {
                return data_masking($content, 'mobile');
            });
            $grid->column('license_no');
            // $grid->column('license_no_pic')->image('', 50, 50);
            // $grid->column('license_no_period');
            // $grid->column('physician_license')->image('', 50, 50);
            // $grid->column('id_card_front_pic')->image('', 50, 50);
            // $grid->column('id_card_back_pic')->image('', 50, 50);
            $grid->column('online_hospital_name');
            $grid->column('department');
            // $grid->column('doctor_title');
            // $grid->column('be_good_at');
            // $grid->column('introduction');
            $grid->column('signed_pic')->image('', 50, 50);
            $grid->column('user.openid', '关联用户');
            $grid->column('created_at');
            // $grid->column('updated_at')->sortable();
            $grid->column('type','医师类型')->using(DoctorModel::DOCTOR_TYPE_MAP);
            // 快捷搜索
            $grid->quickSearch(['name', 'mobile', 'id_card'])->placeholder('请输入[姓名|手机号|身份证号码]')->width(25);

            $grid->filter(function (Grid\Filter $filter) {
                $filter->panel(); // 更改为 panel 布局
                $filter->expand(); // 默认展开搜索框

                $filter->like('name')->width(3);
                $filter->like('id_card')->width(3);
                $filter->like('mobile')->width(3);
                $filter->in('status')->checkbox(DoctorModel::STATUS_MAP)->width(3);
                $filter->in('is_auto', '自动开方')->checkbox(DoctorModel::IS_AUTO_MAP)->width(3);
            });

            $grid->setActionClass(Grid\Displayers\Actions::class);
            // 行按钮控制
            $grid->disableDeleteButton(); // 禁用删除按钮
            $grid->disableViewButton(); // 禁用详情按钮

            // 工具栏按钮控制
            $grid->disableBatchDelete(); // 禁用批量删除
        });
    }

    /**
     * Make a show builder.
     *
     * @param  mixed  $id
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new DoctorRepository(), function (Show $show) {
            $show->field('id')->width(4);
            $show->field('name')->width(4);
            $show->field('id_card')->width(4);
            $show->field('license_no')->width(4);
            $show->field('license_no_pic')->width(4)->image('', 50, 50);
            $show->field('license_no_period')->width(4);
            $show->field('physician_license')->width(4)->image('', 50, 50);
            $show->field('id_card_front_pic')->width(4)->image('', 50, 50);
            $show->field('id_card_back_pic')->width(4)->image('', 50, 50);
            $show->field('online_hospital_name')->width(4);
            $show->field('department')->width(4);
            $show->field('doctor_title')->width(4);
            $show->field('be_good_at')->width(4);
            $show->field('introduction')->width(4);
            $show->field('status')->width(4);
            $show->field('status')->width(4);
            $show->field('created_at')->width(4);
            $show->field('updated_at')->width(4);
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        return Form::make(new DoctorRepository(), function (Form $form) {
            $form->column(6, function (Form $form) {
                $form->display('id');
                $form->text('name')->required()->maxLength(32, '最多输入32个字符');
                $form->mobile('mobile')->required()->rules(function (Form $form) {
                    // 如果不是编辑状态，则添加字段唯一验证
                    if (! $id = $form->model()->id) {
                        return 'unique:App\Models\DoctorModel,mobile';
                    }
                });
                $form->text('id_card')->required();
                $form->text('license_no')->required()->maxLength(30, '最多输入30个字符');
                $form->text('license_no_period');
                $form->text('online_hospital_name')->required()->maxLength(128, '最多输入128个字符');
                $form->text('department')->required()->required()->maxLength(128, '最多输入128个字符');
                $form->text('doctor_title')->required()->maxLength(64, '最多输入64个字符');
                $form->textarea('be_good_at')->required();
                $form->textarea('introduction')->required();
                $form->display('user_id');
                $form->switch('status');
                $form->switch('is_auto');
                $form->radio('type','医师类型')->options(DoctorModel::DOCTOR_TYPE_MAP)->default(DoctorModel::DOCTOR_TYPE_WM)->required();
                $form->display('created_at');
                $form->display('updated_at');
            });
            $form->column(6, function (Form $form) {
                $form->image('license_no_pic')->accept('jpg,png,jpeg')->uniqueName()->autoUpload()->retainable()->removable(false)->required();
                $form->image('physician_license')->accept('jpg,png,jpeg')->uniqueName()->autoUpload()->retainable()->removable(false)->required();
                $form->image('id_card_front_pic')->accept('jpg,png,jpeg')->uniqueName()->autoUpload()->retainable()->removable(false)->required();
                $form->image('id_card_back_pic')->accept('jpg,png,jpeg')->uniqueName()->autoUpload()->retainable()->removable(false)->required();
                $form->image('signed_pic')->accept('jpg,png,jpeg')->uniqueName()->autoUpload()->retainable()->removable(false)->help("<span style='color:red;'>图片尺寸要求：600x400，即3:2的比例</span>");
            });

            $form->saving(function (Form $form) {
                if ($form->mobile && ! preg_match('/^1[3456789]{1}[0-9]{9}$/', $form->mobile)) {
                    return $form->response()->error('手机号格式错误');
                }
                $idValidator = new IdValidator();
                //if ($form->id_card && ! $idValidator->isValid($form->id_card)) {
                if ((! empty($form->id_card)) && strlen($form->id_card) != 18) {
                    return $form->response()->error('身份证格式错误');
                }
            });

            // 右上角按钮控制
            $form->disableDeleteButton(); // 去掉删除按钮
            $form->disableViewButton(); // 去掉跳转详情页按钮
        });
    }
}
