<?php

namespace App\Services;

use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class SmsService
{
    protected $username;

    protected $password;

    protected $apiUrl;

    public function __construct()
    {
        $this->username = config('sms.sms.username');
        $this->password = config('sms.sms.password');
        $this->apiUrl = 'https://www.sms-cly.cn/v7/msg/submit.json';
    }

    /**
     * 发送短信
     *
     * @param  string  $mobile  手机号码，多个号码用半角逗号分隔
     * @param  string  $templateName  模板名称
     * @param  array  $templateData  模板中的变量数据
     * @param  string|null  $seqid  客户自定义消息ID
     * @param  string|null  $dstime  定时时间，格式：yyyy-MM-dd HH:mm:ss
     * @param  string|null  $ext  用户自定义扩展
     * @return array 返回接口的响应
     */
    public function sendSms($mobile, $templateName, $templateData = [], $seqid = null, $dstime = null, $ext = null)
    {
        // 获取模板内容并替换变量
        $content = $this->getFormattedContent($templateName, $templateData);

        if (! $content) {
            return ['resultCode' => '0', 'resultMsg' => '无效的模板名称'];
        }

        // 生成签名
        $sign = md5($this->username.$this->password.$mobile.$content);

        // 构造请求数据
        $payload = [
            'userName' => $this->username,
            'sign' => $sign,
            'mobile' => $mobile,
            'content' => $content,
            'seqid' => $seqid,
            'dstime' => $dstime,
            'ext' => $ext,
        ];
        // 记录短信日志
        Log::info($this->apiUrl, $payload);
        // 发送 HTTP 请求
        $response = Http::withoutVerifying()->post($this->apiUrl, $payload);

        return $response->json();
    }

    /**
     * 获取格式化的模板内容
     *
     * @param  string  $templateName  模板名称
     * @param  array  $templateData  模板数据
     * @return string|null 返回替换变量后的内容
     */
    protected function getFormattedContent($templateName, $templateData)
    {
        // 从配置文件中获取模板
        $template = Config::get("sms.templates.{$templateName}");

        if (! $template) {
            return null; // 如果模板不存在，返回null
        }

        // 使用 Laravel 的 str_replace_array 函数替换变量
        foreach ($templateData as $key => $value) {
            $template = str_replace(":{$key}", $value, $template);
        }

        return $template;
    }
}
