<?php

namespace App\Admin\Controllers;

use App\Models\DosageModel;
use App\Models\PatientModel;
use App\Models\PrescriptionModel;
use Dcat\Admin\Admin;
use Dcat\Admin\Http\Controllers\AdminController;
use Dcat\Admin\Layout\Content;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Laravel\Facades\Image;

// 药店-处方打印
class PrescriptionPrintController extends AdminController
{
    // 处方打印首页，显示输入框，可以搜索处方
    public function index(Content $content): Content
    {
        // 权限判断和数据过滤
        if (! Admin::user()->isRole('pharmacy')) {
            admin_exit(Content::make()->withError(trans('admin.deny')));
        }

        // 加载处方打印页面
        return $content->header('处方打印')
            ->description('根据处方编号搜索并打印处方')
            ->body(view('admin.prescription-print'));
    }

    /**
     * 根据处方编号搜索处方并生成图片
     */
    public function search()
    {
        if (! Admin::user()->isRole('pharmacy')) {
            return response()->json([
                'status' => false,
                'message' => '您没有权限~',
            ]);
        }
        $prescriptionNo = request('prescription_no');
        $dosageData = request('dosage_custom'); // [{"pharmacy_drug_id":6,"dosage_desc":"一天3次1111"},{"pharmacy_drug_id":7,"dosage_desc":"2222每天一次"}]
        $dosageData = json_decode($dosageData, true);
        $isEseal = request('is_eseal', 0); // 是否打印电子印章，实时生成

        if (empty($prescriptionNo)) {
            return response()->json([
                'status' => false,
                'message' => '请输入处方编号~',
            ]);
        }

        try {
            // 获取处方信息
            $prescription = PrescriptionModel::where('id', $prescriptionNo)->where('status', PrescriptionModel::PRESCRIPTION_STATUS_SUCCESS)->first();
            if (! $prescription) {
                return response()->json(['status' => false, 'message' => '未找到该处方或还未审方~']);
            }
            // 检查处方图片是否已经生成
            // if ($isEseal && $prescription->prescription_pic_eseal) {
            //     return response()->json([
            //         'status' => true,
            //         'data' => [
            //             'img_url' => $prescription->prescription_pic_eseal,
            //             'drug_info' => $prescription->drug_info,
            //         ],
            //     ]);
            // } elseif ($prescription->prescription_pic) {
            //     return response()->json([
            //         'status' => true,
            //         'data' => [
            //             'img_url' => $prescription->prescription_pic,
            //             'drug_info' => $prescription->drug_info,
            //         ],
            //     ]);
            // }

            // 创建处方图片
            // 处方背景图宽1653，高2339
            $img = Image::read(public_path('static/images/chufangdan.jpg'));

            $imageWidth = 1653;
            $centerX = $imageWidth / 2; // 手动调整 X 坐标以居中
            // 医院
            $img->text("{$prescription->doctor_online_hospital_name}处方笺", $centerX, 70, function ($font) {
                $font->filename(public_path('static/fonts/SimHei.ttf'));
                $font->size(72);
                $font->color('#000000');
                $font->align('center'); // 使用 align('center') 以确保文本相对于 X 坐标居中
                $font->valign('top');   // 确保文本垂直对齐方式
            });
            // 处方单编号
            $img->text("处方编号：{$prescription->id}", 1170, 190, function ($font) {
                $font->filename(public_path('static/fonts/SimHei.ttf'));
                $font->size(36);
                $font->color('#000000');
            });
            // 姓名
            $img->text($prescription->patient_name, 410, 260, function ($font) {
                $font->filename(public_path('static/fonts/SimHei.ttf'));
                $font->size(36);
                $font->color('#000000');
            });
            // 性别
            $gender = PatientModel::SEX_MAP[$prescription->patient_gender];
            $img->text($gender, 890, 260, function ($font) {
                $font->filename(public_path('static/fonts/SimHei.ttf'));
                $font->size(36);
                $font->color('#000000');
            });
            // 年龄
            $img->text("{$prescription->patient_age}岁", 1270, 260, function ($font) {
                $font->filename(public_path('static/fonts/SimHei.ttf'));
                $font->size(36);
                $font->color('#000000');
            });
            // 开方日期
            $date = \Carbon\Carbon::parse($prescription->prescription_at)->format('Y-m-d');
            $img->text($date, 1215, 330, function ($font) {
                $font->filename(public_path('static/fonts/SimHei.ttf'));
                $font->size(36);
                $font->color('#000000');
            });
            // 诊断
            $img->text($prescription->diagnosis_name, 330, 460, function ($font) {
                $font->filename(public_path('static/fonts/SimHei.ttf'));
                $font->size(32);
                $font->color('#000000');
            });

            $medicines = $prescription->drug_info; // 每个元素包含 'pharmacy_drug', 'num'
            // 根据传过来的用法用量进行修改替换
            if ($dosageData) {
                $dosageMap = [];
                foreach ($dosageData as $dosage) {
                    $dosageMap[$dosage['pharmacy_drug_id']] = $dosage['dosage_id'];
                }
                // 更新 drug_info 中的 dosage_desc
                foreach ($medicines as &$medicine) {
                    if (isset($dosageMap[$medicine['pharmacy_drug_id']])) {
                        $medicine['dosage_id'] = $dosageMap[$medicine['pharmacy_drug_id']];
                        $tmpDosage = DosageModel::where('pharmacy_id', $prescription->pharmacy_id)->where('id', $dosageMap[$medicine['pharmacy_drug_id']])->first();
                        $medicine['dosage_desc'] = $tmpDosage->dosage_desc;
                    }
                }
                // 重新保存处方信息
                $prescription->drug_info = $medicines; // 更新药品信息
                $prescription->save(); // 保存更新后的处方
                $prescription->refresh(); // 刷新处方数据
            }
            // 初始 Y 坐标
            $medicines = $prescription->drug_info;
            $yCoordinate = 650; // 根据需要调整初始 Y 坐标
            foreach ($medicines as $medicine) {
                // 打印药品名称、规格和盒数
                $medicineText = "{$medicine['drug_name']}     {$medicine['spec']}      {$medicine['num']}{$medicine['unit']}";
                $img->text($medicineText, 450, $yCoordinate, function ($font) {
                    $font->filename(public_path('static/fonts/SimHei.ttf'));
                    $font->size(36);
                    $font->color('#000000');
                    $font->align('left');
                });

                // 增加 Y 坐标以打印下一行
                $yCoordinate += 40; // 根据字体大小调整行间距

                // 打印 sig 信息
                $img->text('Sig：  '.$medicine['dosage_desc'], 450, $yCoordinate, function ($font) {
                    $font->filename(public_path('static/fonts/SimHei.ttf'));
                    $font->size(36);
                    $font->color('#000000');
                    $font->align('left');
                });

                // 增加 Y 坐标以准备打印下一个药品
                $yCoordinate += 60; // 根据字体大小调整行间距
            }

            // 打印两行空白
            $yCoordinate += 30; // 空白行间距
            $yCoordinate += 30; // 空白行间距

            // 打印“以下空白”文字
            $img->text('以下空白', 650, $yCoordinate, function ($font) {
                $font->filename(public_path('static/fonts/SimHei.ttf'));
                $font->size(32);
                $font->color('#000000');
                $font->align('center');
            });

            // 医师签名图片
            if ($prescription->doctor_signed_pic) {
                $pharmacistSignPath = file_get_contents($prescription->doctor_signed_pic);
                $doctorSign = Image::read($pharmacistSignPath);
                $doctorSign->resize(150, 100);
                $img->place($doctorSign, 'bottom-left', 400, 410);
            }
            // 药师签名图片
            if ($prescription->pharmacist_signed_pic) {
                $pharmacistSignPath = file_get_contents($prescription->pharmacist_signed_pic);
                $pharmacistSign = Image::read($pharmacistSignPath);
                $pharmacistSign->resize(150, 100);
                $img->place($pharmacistSign, 'bottom-left', 870, 410);
            }
            // 不带印章处方图片生成
            $picName = $prescriptionNo.'.jpg';
            $tempPath = storage_path('app/public').$picName;
            $img->save($tempPath); // 不带电子印章
            // 上传到腾讯云
            Storage::putFileAs('prescriptions', $tempPath, $picName);

            // 电子印章版处方图片生成
            $eseal = Image::read(public_path('static/images/dianzizhang.png'));
            // $eseal->rotate(90);
            $eseal->resize(420, 420);
            $img->place($eseal, 'bottom-right', 180, 500);
            $picEsealName = $prescriptionNo.'-eseal.jpg';
            $tempEsealPath = storage_path('app/public').$picEsealName;
            $img->save($tempEsealPath);
            // 上传到腾讯云
            Storage::putFileAs('prescriptions', $tempEsealPath, $picEsealName);

            // 删除临时文件
            unlink($tempPath);
            unlink($tempEsealPath);

            $cosPath = "prescriptions/{$picName}";
            $cosEsealPath = "prescriptions/{$picEsealName}";

            // 将生成的处方图片路径存储到数据库
            $prescription->prescription_pic = $cosPath;
            $prescription->prescription_pic_eseal = $cosEsealPath;
            $prescription->save(); // 保存处方图片路径

            if ($isEseal) {
                $imgUrl = Storage::url($cosEsealPath);
            } else {
                $imgUrl = Storage::url($cosPath);
            }

            $dosageList = DosageModel::where('pharmacy_id', $prescription->pharmacy_id)->select('id', 'dosage_desc')->get()->toArray();

            return response()->json([
                'status' => true,
                'data' => [
                    'img_url' => $imgUrl,
                    'drug_info' => $prescription->drug_info,
                    'dosage_list' => $dosageList,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => '处方生成失败:'.$e->getMessage(),
            ]);
        }
    }
}
