<?php

namespace App\Api\Controllers;

use App\Http\Controllers\BaseApiController;
use App\Models\PharmacistModel;
use App\Models\PharmacyModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class PharmacistController extends BaseApiController
{
    // 药师列表
    public function pharmacistList()
    {
        $authInfo = auth('api')->user();
        $pharmacy = PharmacyModel::where('user_id', $authInfo->id)->first();
        if (! $pharmacy) {
            return $this->failed('请退出后重新登录!');
        }
        $pharmacist = PharmacistModel::where('pharmacy_id', $pharmacy->id)->get();

        return $this->success($pharmacist);
    }

    // 药师信息
    public function detail(Request $request)
    {
        $pharmacist_id = $request->input('pharmacist_id');
        if (empty($pharmacist_id) || ! filter_var($pharmacist_id, FILTER_VALIDATE_INT)) {
            return $this->failed('ID不能为空且必须为整数');
        }
        $authInfo = auth('api')->user();
        // 获取药店信息
        $pharmacy = PharmacyModel::where('user_id', $authInfo->id)->first();
        if (! $pharmacy) {
            return $this->failed('请退出后重新登录!');
        }
        $pharmacist = PharmacistModel::where('pharmacy_id', $pharmacy->id)->where('id', $pharmacist_id)->first();
        if (! $pharmacist) {
            return $this->failed('该药师不存在');
        } else {
            return $this->success($pharmacist);
        }
    }

    // 证书上传
    public function uploadCertificate(Request $request)
    {

        // 验证上传的图片文件
        $validator = Validator::make($request->all(), [
            'image' => 'required|image|mimes:jpeg,png,jpg|max:10240',
        ]);
        if ($validator->fails()) {
            return $this->failed('图片必须2Mb以内，且是jpeg,png,jpg三种格式~');
        }

        // 验证药师编号
        $authInfo = auth('api')->user();
        // 获取药店信息
        $pharmacy = PharmacyModel::where('user_id', $authInfo->id)->first();
        if (empty($pharmacy)) {
            return $this->failed('该药店不存在');
        }

        // 获取上传的图片
        if ($request->hasFile('image')) {
            $path = Storage::putFile('license-images', $request->file('image'));
            // 返回完整图片地址
            $imageUrl = Storage::url($path);

            return $this->success(['image_id' => $path, 'full_url' => $imageUrl]);
        } else {
            return $this->failed('请上传图片~');
        }
    }

    // 药师新增
    public function add(Request $request)
    {
        $authInfo = auth('api')->user();
        // 获取药店信息
        $pharmacy = PharmacyModel::where('user_id', $authInfo->id)->first();
        $pharmacist = new PharmacistModel();
        $pharmacist->pharmacy_id = $pharmacy->id;
        $pharmacist->name = str_replace(' ', '', $request->input('name'));
        $pharmacist->id_card = str_replace(' ', '', $request->input('id_card'));
        $pharmacist->mobile = str_replace(' ', '', $request->input('mobile'));
        $pharmacist->license_number = str_replace(' ', '', $request->input('license_number'));
        $pharmacist->practicing_license = $request->input('practicing_license'); // 执业注册证书链接
        $pharmacist->practicing_license_expired_time = $request->input('practicing_license_expired_time');
        $pharmacist->physician_license = $request->input('physician_license'); // 执业资格证书链接
        $pharmacist->status = 0;
        if ($pharmacist->save()) {
            return $this->success($pharmacist);
        }

        return $this->failed('药师新增失败！');
    }

    public function scanAdd(Request $request)
    {
        $authInfo = auth('api')->user();
        // 获取药店信息
        $pharmacy = PharmacyModel::where('user_id', $authInfo->id)->first();

        if (! $pharmacy) {
            return $this->failed('未找到药店信息！');
        }
        // 检查是否传递了药师的id
        $pharmacistId = $request->input('id');
        if ($pharmacistId) {
            // 查询并复制已有药师数据
            $existingPharmacist = PharmacistModel::where('id', $pharmacistId)->first();
            if (! $existingPharmacist) {
                return $this->failed('未找到对应药师信息！');
            }
            $pharmacist = $existingPharmacist->replicate();
            // 设置药师属性
            $pharmacist->pharmacy_id = $pharmacy->id;
            $pharmacist->name = $pharmacist->name ?? '';
            $pharmacist->id_card = $pharmacist->id_card ?? '';
            $pharmacist->mobile = $pharmacist->mobile ?? '';
            $pharmacist->license_number = $pharmacist->license_number ?? '';
            $pharmacist->practicing_license = $pharmacist->practicing_license ?? '';
            $pharmacist->practicing_license_expired_time = $pharmacist->practicing_license_expired_time ?? null;
            $pharmacist->physician_license = $pharmacist->physician_license ?? '';
            $pharmacist->status = 0;
            // 保存药师数据
            if ($pharmacist->save()) {
                return $this->success($pharmacist);
            }
        }

        return $this->failed('药师新增失败！');
    }

    // 药师编辑
    public function update(Request $request)
    {
        $authInfo = auth('api')->user();
        // 获取药店信息
        $pharmacy = PharmacyModel::where('user_id', $authInfo->id)->first();
        if (! $pharmacy) {
            return $this->failed('该药店不存在');
        }
        $pharmacist = PharmacistModel::where('id', $request->input('id'))->where('pharmacy_id', $pharmacy->id)->first();
        if (empty($pharmacist)) {
            return $this->failed('该药师不存在');
        }
        $practicing_license = $request->input('practicing_license');
        $practicing_license_path = '';
        if( !empty($practicing_license) ){
            $practicing_licenses = parse_url($practicing_license);
            $practicing_license_path = $practicing_licenses['path'] ?? '';
        }

        $physician_license = $request->input('physician_license');
        $physician_license_path = '';
        if( !empty($physician_license) ){
            $physician_licenses = parse_url($physician_license);
            $physician_license_path = $physician_licenses['path'] ?? '';
        }

        $pharmacist->name = str_replace(' ', '', $request->input('name'));
        $pharmacist->id_card = str_replace(' ', '', $request->input('id_card'));
        $pharmacist->mobile = str_replace(' ', '', $request->input('mobile'));
        $pharmacist->license_number = str_replace(' ', '', $request->input('license_number'));
        $pharmacist->practicing_license = $practicing_license_path; // 执业注册证书链接
        $pharmacist->practicing_license_expired_time = $request->input('practicing_license_expired_time');
        $pharmacist->physician_license = $physician_license_path; // 执业资格证书链接

        if ($pharmacist->save()) {
            return $this->success($pharmacist);
        }

        return $this->failed('药师编辑失败');
    }

    // 设置默认药师
    public function setDefault(Request $request)
    {
        $authInfo = auth('api')->user();
        // 获取药店信息
        $pharmacy = PharmacyModel::where('user_id', $authInfo->id)->first();
        if (! $pharmacy) {
            return $this->failed('该药店不存在');
        }
        // 把该药店下的所有药师默认状态改为0
        PharmacistModel::where('pharmacy_id', $pharmacy->id)->update(['is_default' => 0]);
        $pharmacist = PharmacistModel::where('id', $request->input('id'))->first();
        if (empty($pharmacist)) {
            return $this->failed('该药师不存在');
        }
        $pharmacist->is_default = 1;
        if ($pharmacist->save()) {
            return $this->success($pharmacist);
        }

        return $this->failed('设置默认药师失败!');
    }

    // 药师签名上传
    public function upload(Request $request)
    {

        // 验证上传的图片文件
        $validator = Validator::make($request->all(), [
            'image' => 'required|image|mimes:jpeg,png,jpg|max:10240',
        ]);
        if ($validator->fails()) {
            return $this->failed('图片必须2Mb以内，且是jpeg,png,jpg三种格式~');
        }
        // 验证药师编号
        $pharmacist_id = $request->input('pharmacist_id');
        if (empty($pharmacist_id) || ! filter_var($pharmacist_id, FILTER_VALIDATE_INT)) {
            return $this->failed('ID不能为空且必须为整数');
        }
        $authInfo = auth('api')->user();
        // 获取药店信息
        $pharmacy = PharmacyModel::where('user_id', $authInfo->id)->first();
        if (empty($pharmacy)) {
            return $this->failed('该药店不存在');
        }
        // 获取药师信息
        $pharmacist = PharmacistModel::where('id', $pharmacist_id)->where('pharmacy_id', $pharmacy->id)->first();
        if (empty($pharmacist)) {
            return $this->failed('该药师不存在');
        }

        // 获取上传的图片
        if ($request->hasFile('image')) {
            $path = Storage::putFile('license-images', $request->file('image'));
            // 返回完整图片地址
            $imageUrl = Storage::url($path);

            // 签名图片地址记录到数据库
            $pharmacist->signed_pic = $path;
            if ($pharmacist->save()) {
                return $this->success(['message' => 'ok', 'url' => $imageUrl]);
            }

            return $this->failed('签名图片上传失败');
        } else {
            return $this->failed('签名图片上传失败');
        }
    }

    // 药师删除
    public function delete(Request $request)
    {
        $authInfo = auth('api')->user();
        // 获取药店信息
        $pharmacy = PharmacyModel::where('user_id', $authInfo->id)->first();
        if (! $pharmacy) {
            return $this->failed('该药店不存在');
        }
        // 判断该药店下是否有几个开启的药师
        $pharmacist = PharmacistModel::where('pharmacy_id', $pharmacy->id)->where('status', 1)->count();
        if ($pharmacist <= 1) {
            return $this->failed('无法删除，至少需要一个开启的药师！');
        }

        $id = $request->input('id');
        $data = PharmacistModel::where('id', $id)->where('pharmacy_id', $pharmacy->id)->first();
        if (! $data) {
            return $this->failed('该药师不存在');
        }
        if ($data->delete()) {
            return $this->success($data);
        }

        return $this->failed('删除失败');
    }
}
