<?php

if (! function_exists('object_2_array')) {
    // 对象转换成数组
    function object_2_array($object)
    {
        return array_map('get_object_vars', $object);
    }
}

if (! function_exists('clean_ids')) {
    /**
     * 清理输入字符串，提取数字ID，并返回一个整数数组或以逗号分隔的字符串。
     *
     * @param  string  $idsString  原始输入字符串，包含多个ID和不同的分隔符
     * @param  bool  $returnArray  是否返回数组，默认为 false，返回字符串
     * @return array|string 返回整数ID数组或以逗号分隔的ID字符串
     */
    function clean_ids($idsString, $returnArray = false)
    {
        // 替换所有非数字字符为英文逗号
        $ids = preg_replace('/[\D]+/', ',', trim($idsString));

        // 移除字符串开头和结尾的逗号
        $ids = trim($ids, ',');

        if (! $returnArray) {
            // 返回处理后的字符串
            return $ids;
        }

        // 分割字符串为数组，并过滤掉空字符串
        $idArr = array_filter(explode(',', $ids), 'strlen');

        // 将数组的每个元素转换为整数并去重
        $idArr = array_unique($idArr);

        return $idArr;
    }
}

if (! function_exists('getAgeByIdCard')) {
    function getAgeByIdCard($idCard)
    {
        // 验证身份证号码的长度
        if (strlen($idCard) != 18) {
            return ''; // 未知
        }

        // 提取出生日期
        $birthDate = substr($idCard, 6, 8); // 身份证的第7到第14位为出生日期
        $year = (int) substr($birthDate, 0, 4);
        $month = (int) substr($birthDate, 4, 2);
        $day = (int) substr($birthDate, 6, 2);

        // 获取当前日期
        $currentDate = new DateTime();
        $birthDateTime = new DateTime("$year-$month-$day");

        // 计算年龄
        $age = $currentDate->diff($birthDateTime)->y;

        return $age;
    }
}

if (! function_exists('getSexByIdCard')) {
    /**
     * 根据身份证获取性别
     *
     * @param  string  $idCard  身份正号
     * @return int 1=男, 2=女
     */
    function getSexByIdCard($idCard)
    {
        // 验证身份证号码的长度
        if (strlen($idCard) != 18) {
            return 0; // 未知
        }

        // 提取性别位（第17位）
        $gender = (int) substr($idCard, 16, 1); // 第17位的性别数字

        // 判断性别
        if ($gender % 2 == 0) {
            return 2; // 偶数女
        } else {
            return 1; // 基数男
        }
    }
}

if (! function_exists('data_masking')) {
    /**
     * 数据脱敏.
     *
     * @param  string|null  $strType  脱敏类型，姓名、生日、手机号……
     * @param  string  $str  被脱敏文本
     * @return string 脱敏后文本
     */
    function data_masking($str, string $strType)
    {
        if (is_null($str) || $str === '' || trim($str) === '') {
            return $str;
        }
        switch ($strType) {
            case 'uname': // 如果是字符和就显示前后各一位，如果是手机号则展示前3位和后4位
                if (preg_match('/^1[3456789]{1}[0-9]{9}$/', $str)) {
                    $maskStr = substr_replace($str, '****', 3, 4);
                } else {
                    //普通账号不做脱敏展示
                    $maskStr = $str;
                    //$maskStr = str_repeat("*", mb_strlen($str)-1) . mb_substr($str, -1, 1);
                }
                break;
            case 'truename': // 只隐藏第一位，浙里办
                $maskStr = '*'.mb_substr($str, 1);
                break;
            case 'truename2': // 只隐藏第二位，其他
                $len = mb_strlen($str);
                if ($len == 2) {
                    $maskStr = mb_substr($str, 0, 1).'*';
                }
                if ($len < 2) {
                    $maskStr = $str;
                }
                if ($len > 2) {
                    $maskStr = mb_substr($str, 0, 1).str_repeat('*', mb_strlen($str) - 2).mb_substr($str, -1, 1);
                }
                break;
            case 'birthday': // 仅展示最后一位
                $maskStr = '****-**-*'.substr($str, -1, 1);
                break;
            case 'bankcard': // 展示前4位和后3位
                $maskStr = substr($str, 0, 4).'************'.substr($str, -4, 3);
                break;
            case 'idcard': // 只展示前6位和后4位
                $maskStr = substr($str, 0, 6).str_repeat('*', strlen($str) - 10).substr($str, -4);
                break;
            case 'mobile': // 展示前3位和后4位
                $maskStr = substr_replace($str, '****', 3, 4);
                break;
            case 'email': // 名字部分只展示首位和末位
                $pos = stripos($str, '@', 0);
                $maskStr = substr($str, 0, 1).str_repeat('*', $pos - 2).substr($str, $pos - 1, 1).substr($str, $pos);
                break;
            case 'address': // 地址的详细地址部分
                $maskStr = '******';
                break;
            case 'detailedaddress':// 详细地址，只展示前6位
                $maskStr = mb_substr($str, 0, 6).'******';
                break;
            case 'image':
                $maskStr = env('DEFAULT_PIC');
                break;
            case 'sex':
                $maskStr = '*';
                break;
            default:
                $maskStr = $str;
                break;
        }

        return $maskStr;
    }
}

if (! function_exists('validateBirthday')) {
    function validateBirthday($birthday)
    {
        $year = substr($birthday, 0, 4);
        $month = substr($birthday, 4, 2);
        $day = substr($birthday, 6, 2);
        $timestamp = mktime(0, 0, 0, $month, $day, $year);

        return $timestamp !== false && $timestamp < strtotime('midnight');
    }

}

if (! function_exists('validateIDCard')) {

    function validateIDCard($idCard) {
        // 长度检查
        if (!preg_match('/^\d{15}$|^\d{17}[\dXx]$/', $idCard)) {
            return false;
        }

        // 15位转18位（如果是15位身份证）
        if (strlen($idCard) === 15) {
            $idCard = convertIDCard15to18($idCard);
        }

        // 检查出生日期
        $birthDate = substr($idCard, 6, 8); // 提取出生日期部分
        $year = substr($birthDate, 0, 4);
        $month = substr($birthDate, 4, 2);
        $day = substr($birthDate, 6, 2);

        if (!checkdate((int)$month, (int)$day, (int)$year)) {
            return false;
        }

        // 校验码验证
        return checkIDCardChecksum($idCard);
    }

    function convertIDCard15to18($idCard15) {
        // 15位身份证转换为18位
        $prefix = substr($idCard15, 0, 6) . '19' . substr($idCard15, 6);
        $checksum = calculateChecksum($prefix);
        return $prefix . $checksum;
    }

    function checkIDCardChecksum($idCard18) {
        // 检查身份证校验码
        $base = substr($idCard18, 0, 17);
        $checksum = calculateChecksum($base);
        return strtoupper($checksum) === strtoupper($idCard18[17]);
    }

    function calculateChecksum($base) {
        // 计算校验码
        $weights = [7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2];
        $checksumMap = ['1', '0', 'X', '9', '8', '7', '6', '5', '4', '3', '2'];
        $sum = 0;

        for ($i = 0; $i < 17; $i++) {
            $sum += $base[$i] * $weights[$i];
        }

        return $checksumMap[$sum % 11];
    }
}