<?php

namespace App\Admin\Controllers;

use App\Admin\Extensions\ToolBar\Actions\DrugImportAction;
use App\Admin\Repositories\DrugRepository;
use App\Models\DrugModel;
use App\Models\DrugTagModel;
use App\Models\DrugUnitModel;
use Box\Spout\Common\Entity\Style\Color;
use Box\Spout\Writer\Common\Creator\Style\StyleBuilder;
use Box\Spout\Writer\Common\Creator\WriterEntityFactory;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Http\Controllers\AdminController;
use Dcat\Admin\Show;
use Dcat\EasyExcel\Excel;
use Illuminate\Http\Request;

// 药品池
class DrugController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(new DrugRepository('tag'), function (Grid $grid) {
            $grid->model()->orderBy('id', 'DESC');

            $grid->column('id')->sortable();
            $grid->column('name');
            $grid->column('code');
            $grid->column('product_name');
            $grid->column('spec');
            $grid->column('preparation_pec');
            $grid->column('dosage_form');
            $grid->column('unit');
            $grid->column('factory');
            $grid->column('approval_no');
            $grid->column('mnemonic');
            $grid->column('standard_code');
            $grid->column('category_code');
            $grid->column('category_name');
            $grid->column('is_foreign')->using(DrugModel::FOREIGN_MAP);
            $grid->column('is_rx')->using(DrugModel::RX_MAP);
            $grid->column('is_si')->using(DrugModel::SI_MAP);
            $grid->column('limit_buy_7');
            $grid->column('tag')->display(function ($tags) {
                return collect($tags)->map(function ($tag) {
                    return "<span class='label' style='background:#6d8be6'>{$tag->tag_name}</span>";
                })->implode('&nbsp;');
            });
            $grid->column('product_id');

            // $grid->column('created_at');
            // $grid->column('updated_at')->sortable();

            // 工具栏普通按钮
            $grid->tools(function ($tools) {
                $tools->append(new DrugImportAction()); // 导入发货信息
            });

            // 快捷搜索
            $grid->quickSearch(['name', 'code', 'factory'])->placeholder('请输入[药品名称/药品简码/生产厂家]')->width(25);

            $grid->filter(function (Grid\Filter $filter) {
                $filter->panel(); // 更改为 panel 布局
                $filter->expand(); // 默认展开搜索框

                $filter->like('name')->width(3);
                $filter->like('code')->width(3);
                $filter->like('factory')->width(3);
                $filter->like('standard_code')->width(3);
            });

            $grid->setActionClass(Grid\Displayers\Actions::class);
            // 行按钮控制
            $grid->disableDeleteButton(); // 禁用删除按钮
            $grid->disableViewButton(); // 禁用详情按钮

            // 工具栏按钮控制
            $grid->disableBatchDelete(); // 禁用批量删除

            // 导出按钮
            $grid->export()->filename('药品');
        });
    }

    /**
     * Make a show builder.
     *
     * @param  mixed  $id
     * @return Show
     */
    protected function detail($id) {}

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        return Form::make(new DrugRepository('tag'), function (Form $form) {
            $form->column(6, function (Form $form) {
                $form->display('id');
                $form->display('code');
                $form->text('name')->required()->maxLength(64);
                $form->text('product_name')->maxLength(64);
                $form->text('spec')->required()->maxLength(128);
                $form->text('preparation_pec')->maxLength(128);
                $form->text('dosage_form')->required()->maxLength(128);
                $form->select('unit')->options(DrugUnitModel::all()->pluck('name', 'name'));
                $form->text('factory')->required()->maxLength(128);
                $form->text('approval_no')->required()->maxLength(64);
                $form->text('mnemonic')->maxLength(64);
            });
            $form->column(6, function (Form $form) {
                $form->text('standard_code')->required()->rules(function (Form $form) {
                    // 如果不是编辑状态，则添加字段唯一验证
                    if (! $id = $form->model()->id) {
                        return 'unique:App\Models\DrugModel,standard_code';
                    }
                });
                $form->text('category_code')->maxLength(32);
                $form->text('category_name')->maxLength(32);
                $form->switch('is_foreign');
                $form->switch('is_rx');
                $form->switch('is_si');
                $array = range(1, 10);
                $limitOption = array_combine($array, $array);
                $form->select('limit_buy_7')->placeholder('请选择限购数量')->options($limitOption)->help('限购数量')->required();
                $form->multipleSelect('tag', '标签')
                    ->options(function () {
                        return DrugTagModel::all()->pluck('tag_name', 'id');
                    })
                    ->customFormat(function ($v) {
                        return array_column($v, 'id');
                    });

                $form->display('created_at');
                $form->display('updated_at');
            });
            // 右上角按钮控制
            $form->disableDeleteButton(); // 去掉删除按钮
            $form->disableViewButton(); // 去掉跳转详情页按钮
        });
    }

    /**
     * 药品导入模板
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportDrugTemplate(Request $request)
    {
        $sheet1Head = ['药品名称', '本位码', '单位', '规格', '剂型', '生产厂家', '批准文号', '是否处方药', '是否医保药品', '君元ID'];
        $sheet1Data = [['加味天麻胶囊', '86903692001013', '粒', '36s', '每粒装0.25g', '葵花药业集团(佳木斯)有限公司', '国药准字Z23020970', '是', '是', '8000']];

        $sheet1 = Excel::createSheet($sheet1Data, '药品信息', $sheet1Head)->row(function (array $row) {
            $style = (new StyleBuilder)
                ->setBackgroundColor(Color::YELLOW)
                ->build();

            return WriterEntityFactory::createRowFromArray($row, $style);
        });

        return Excel::export([$sheet1])->headings(false)->download('药品导入模板.xlsx');
    }
}
