<?php

namespace App\Admin\Controllers;

use App\Admin\Repositories\PharmacistRepository;
use App\Models\PharmacistModel;
use App\Models\PharmacyModel;
use Carbon\Carbon;
use Dcat\Admin\Admin;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Http\Controllers\AdminController;
use Dcat\Admin\Show;
use Dcat\Admin\Widgets\Tab;
use Illuminate\Http\Request;
use Jxlwqq\IdValidator\IdValidator;

// 药师
class PharmacistController extends AdminController
{
    public function __construct(Request $request)
    {
        $this->view = $request->view;

        return $this;
    }

    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(new PharmacistRepository('pharmacy'), function (Grid $grid) {
            Admin::style(
                <<<'CSS'
        .nav-tabs {
            background-color: #fff;
            margin-top: 20px;
            box-shadow: 0 2px 4px 0 rgba(0,0,0,.05);
            border-radius: .25rem;
        }
CSS
            );

            // 获取当前日期
            // $now = Carbon::now();
            if (! $this->view || $this->view == 1) {
            } elseif ($this->view == 2) {
                $grid->model()->where('practicing_license_expired_time', '>=', Carbon::now())->where('practicing_license_expired_time', '<=', Carbon::now()->addMonths(1));
            } elseif ($this->view == 3) {
                $grid->model()->where('practicing_license_expired_time', '<', Carbon::now());
            }

            $grid->header(function () {
                $tab = Tab::make();
                $tab->addLink('药师列表', '?view=1', (! $this->view || $this->view == 1) ? true : false);
                $tab->addLink('临期药师列表', '?view=2', $this->view == 2 ? true : false);
                $tab->addLink('失效药师列表', '?view=3', $this->view == 3 ? true : false);

                return $tab;
            });

            $grid->model()->orderBy('id', 'desc');
            $grid->column('id')->sortable();
            $grid->column('status')->switch();
            $grid->column('name');
            $grid->column('id_card')->display(function ($content) {
                return data_masking($content, 'idcard');
            });
            $grid->column('license_number');
            $grid->column('signed_pic')->image('', 50, 50);
            $grid->column('mobile')->display(function ($content) {
                return data_masking($content, 'mobile');
            });
            $grid->column('practicing_license_expired_time');
            $grid->column('pharmacy.name', '关联药店');
            $grid->column('is_default')->using(PharmacistModel::IS_DEFAULT_MAP)->dot(PharmacistModel::IS_DEFAULT_MAP_COLOR);
            // $grid->column('practicing_license')->image('', 50, 50);
            // $grid->column('physician_license')->image('', 50, 50);
            // $grid->column('created_at');
            // $grid->column('updated_at')->sortable();

            // 快捷搜索
            $grid->quickSearch(['name', 'mobile', 'id_card'])->placeholder('请输入[姓名|手机号|身份证号码]')->width(25);

            $grid->filter(function (Grid\Filter $filter) {
                $filter->panel(); // 更改为 panel 布局
                $filter->expand(); // 默认展开搜索框

                $filter->like('name')->width(3);
                $filter->like('id_card')->width(3);
                $filter->like('mobile')->width(3);
                $filter->equal('pharmacy_id')->select(PharmacyModel::all()->pluck('name', 'id'))->width(3);
            });

            $grid->setActionClass(Grid\Displayers\Actions::class);
            // 行按钮控制
            $grid->disableDeleteButton(); // 禁用删除按钮
            $grid->disableViewButton(); // 禁用详情按钮

            // 工具栏按钮控制
            $grid->disableBatchDelete(); // 禁用批量删除
        });
    }

    /**
     * Make a show builder.
     *
     * @param  mixed  $id
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new PharmacistRepository(), function (Show $show) {
            $show->field('id')->width(4);
            $show->field('name')->width(4);
            $show->field('id_card')->width(4);
            $show->field('license_number')->width(4);
            $show->field('mobile')->width(4);
            $show->field('practicing_license')->width(4);
            $show->field('practicing_license_period')->width(4);
            $show->field('physician_license')->width(4);
            $show->field('signed_pic')->width(4);
            $show->field('status')->width(4);
            $show->field('created_at')->width(4);
            $show->field('updated_at')->width(4);

            $show->panel()->tools(function ($tools) {
                $tools->disableDelete(); // 禁止删除按钮
            });
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        return Form::make(new PharmacistRepository(), function (Form $form) {
            $form->column(6, function (Form $form) {
                $form->display('id');
                $form->text('name')->required()->maxLength(32, '最多输入32个字符');
                $form->mobile('mobile')->required()->rules(function (Form $form) {
                    // 如果不是编辑状态，则添加字段唯一验证
                    if (! $id = $form->model()->id) {
                        return 'unique:App\Models\PharmacistModel,mobile';
                    }
                });
                $form->text('id_card')->required()->maxLength(18, '最多输入18个字符');
                $form->text('license_number')->required()->maxLength(32, '最多输入32个字符');
                $form->date('practicing_license_expired_time')->required();
                $form->select('pharmacy_id')->options(PharmacyModel::all()->pluck('name', 'id'))->required();
                $form->switch('status');
                $form->display('created_at');
                $form->display('updated_at');
            });
            $form->column(6, function (Form $form) {
                $form->image('practicing_license')->accept('jpg,png,jpeg')->uniqueName()->autoUpload()->retainable()->removable(false)->required();
                $form->image('physician_license')->accept('jpg,png,jpeg')->uniqueName()->autoUpload()->retainable()->removable(false)->required();
                $form->image('signed_pic')->accept('jpg,png,jpeg')->uniqueName()->autoUpload()->retainable()->removable(false);
            });

            $form->saving(function (Form $form) {
                if ($form->mobile && ! preg_match('/^1[3456789]{1}[0-9]{9}$/', $form->mobile)) {
                    return $form->response()->error('手机号格式错误');
                }
                $idValidator = new IdValidator();
                if ($form->id_card && ! $idValidator->isValid($form->id_card)) {
                    return $form->response()->error('身份证格式错误');
                }
            });

            // 右上角按钮控制
            $form->disableDeleteButton(); // 去掉删除按钮
            $form->disableViewButton(); // 去掉跳转详情页按钮
        });
    }
}
