<?php

namespace App\Admin\Controllers;

use App\Admin\Extensions\ToolBar\Actions\PharmacyDosageImportAction;
use App\Admin\Repositories\DosageRepository;
use Box\Spout\Common\Entity\Style\Color;
use Box\Spout\Writer\Common\Creator\Style\StyleBuilder;
use Box\Spout\Writer\Common\Creator\WriterEntityFactory;
use Dcat\Admin\Admin;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Http\Controllers\AdminController;
use Dcat\Admin\Layout\Content;
use Dcat\Admin\Show;
use Dcat\EasyExcel\Excel;
use Illuminate\Http\Request;

// 用法用量
class DosageController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(new DosageRepository('pharmacy'), function (Grid $grid) {
            // 权限判断和数据过滤
            if (! Admin::user()->pharmacy_id) {
                admin_exit(Content::make()->withError(trans('admin.deny')));
            }

            $grid->model()->orderBy('id', 'DESC');

            $grid->model()->where('pharmacy_id', Admin::user()->pharmacy_id);
            $grid->column('id')->sortable();
            $grid->column('pharmacy.name', '药店');
            $grid->column('dosage_desc');
            $grid->column('dosage_show');

            // 工具栏普通按钮
            $grid->tools(function ($tools) {
                $tools->append(new PharmacyDosageImportAction()); // 导入药品信息
            });

            $grid->filter(function (Grid\Filter $filter) {
                $filter->panel(); // 更改为 panel 布局
                $filter->expand(); // 默认展开搜索框

                $filter->like('dosage_desc')->width(3);
            });

            // 行按钮控制
            // $grid->disableDeleteButton(); // 禁用删除按钮
            $grid->disableViewButton(); // 禁用详情按钮

            // 工具栏按钮控制
            $grid->disableBatchDelete(); // 禁用批量删除
        });
    }

    /**
     * Make a show builder.
     *
     * @param  mixed  $id
     * @return Show
     */
    protected function detail($id)
    {
        admin_exit(Content::make()->withError(trans('admin.deny')));

        return Show::make($id, new DosageRepository(), function (Show $show) {
            $show->field('id');
            $show->field('dosage_desc');
            $show->field('dosage_show');
            $show->field('created_at');
            $show->field('updated_at');

            $show->panel()->tools(function ($tools) {
                $tools->disableDelete(); // 禁止删除按钮
            });
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        return Form::make(new DosageRepository(), function (Form $form) {
            // 权限判断和数据过滤
            if (! Admin::user()->pharmacy_id) {
                admin_exit(Content::make()->withError(trans('admin.deny')));
            }
            if ($form->isEditing() && (Admin::user()->pharmacy_id != $form->model()->pharmacy_id)) {
                admin_exit(Content::make()->withError(trans('admin.deny')));
            }
            $form->display('id')->width(4);
            $form->hidden('pharmacy_id');
            $form->text('dosage_desc')->width(4)->required()->required()->maxLength(128, '最多输入128个字符');
            $form->textarea('dosage_show')->width(4)->required();

            $form->display('created_at')->width(4);
            $form->display('updated_at')->width(4);

            $form->saving(function (Form $form) {
                // 不能动
                $form->pharmacy_id = Admin::user()->pharmacy_id;
            });
            // 右上角按钮控制
            $form->disableDeleteButton(); // 去掉删除按钮
            $form->disableViewButton(); // 去掉跳转详情页按钮
        });
    }

    /**
     * 药品导入模板
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportDosageTemplate(Request $request)
    {
        $sheet1Head = ['用法用量', '显示内容'];
        $sheet1Data = [['220ug qd 每鼻孔2喷', '220ug qd 每鼻孔2喷']];

        $sheet1 = Excel::createSheet($sheet1Data, '用法用量信息', $sheet1Head)->row(function (array $row) {
            $style = (new StyleBuilder)
                ->setBackgroundColor(Color::YELLOW)
                ->build();

            return WriterEntityFactory::createRowFromArray($row, $style);
        });

        return Excel::export([$sheet1])->headings(false)->download('用法用量导入模板.xlsx');
    }
}
