<?php

namespace App\Admin\Controllers;

use App\Admin\Repositories\PatientRepository;
use App\Models\PatientModel;
use App\Models\PharmacyModel;
use Dcat\Admin\Form;
use Dcat\Admin\Grid;
use Dcat\Admin\Http\Controllers\AdminController;
use Dcat\Admin\Show;

// 问诊人
class PatientController extends AdminController
{
    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        return Grid::make(new PatientRepository(['user', 'pharmacy']), function (Grid $grid) {
            $grid->model()->orderBy('id', 'DESC');

            $grid->column('id')->sortable();
            $grid->column('name');
            $grid->column('id_card')->display(function ($content) {
                return data_masking($content, 'idcard');
            });
            $grid->column('gender', '性别')->using(PatientModel::SEX_MAP);
            $grid->column('age', '年龄')->display(function ($value) {
                return getAgeByIdCard($this->id_card);
            });
            $grid->column('mobile')->display(function ($content) {
                return data_masking($content, 'mobile');
            });
            $grid->column('is_default')->using([0 => '否', 1 => '是']);
            $grid->column('user.id', '关联用户');
            $grid->column('pharmacy.name', '关联药店');

            // 快捷搜索
            $grid->quickSearch(['name', 'id_card', 'mobile'])->placeholder('请输入[姓名/身份证/手机号]')->width(25);

            $grid->filter(function (Grid\Filter $filter) {
                $filter->panel(); // 更改为 panel 布局
                $filter->expand(); // 默认展开搜索框

                $filter->like('name')->width(3);
                $filter->like('id_card')->width(3);
                $filter->like('mobile')->width(3);
                $filter->equal('pharmacy_id')->select(PharmacyModel::all()->pluck('name', 'id'))->width(3);
            });

            $grid->setActionClass(Grid\Displayers\Actions::class);
            // 行按钮控制
            $grid->disableViewButton(); // 禁用详情按钮
            $grid->disableCreateButton(); // 禁用创建按钮
            $grid->disableDeleteButton(); // 禁用删除按钮

            // 工具栏按钮控制
            $grid->disableBatchDelete(); // 禁用批量删除
        });
    }

    /**
     * Make a show builder.
     *
     * @param  mixed  $id
     * @return Show
     */
    protected function detail($id)
    {
        return Show::make($id, new PatientRepository(), function (Show $show) {
            $show->field('id')->width(4);
            $show->field('name')->width(4);
            $show->field('gender')->width(4);
            $show->field('id_card')->width(4);
            $show->field('mobile')->width(4);
            $show->field('created_at')->width(4);
            $show->field('updated_at')->width(4);

            $show->panel()->tools(function ($tools) {
                $tools->disableDelete(); // 禁止删除按钮
            });
        });
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        return Form::make(new PatientRepository(), function (Form $form) {
            $form->display('id')->width(4);
            $form->text('name')->width(4)->required()->maxLength(32, '最多输入32个字符');
            $form->text('id_card')->width(4)->required()->maxLength(18, '最多输入18个字符');
            $form->mobile('mobile')->width(4)->required();
            $form->display('user_id')->width(4);
            $form->select('pharmacy_id')->options(PharmacyModel::all()->pluck('name', 'id'))->disable()->width(4);

            $form->display('created_at')->width(4);
            $form->display('updated_at')->width(4);

            $form->saving(function (Form $form) {
                if ($form->mobile && ! preg_match('/^1[3456789]{1}[0-9]{9}$/', $form->mobile)) {
                    return $form->response()->error('手机号格式错误');
                }
                $idValidator = new IdValidator();
                if ($form->id_card && ! $idValidator->isValid($form->id_card)) {
                    return $form->response()->error('身份证格式错误');
                }
            });

            // 右上角按钮控制
            $form->disableDeleteButton(); // 去掉删除按钮
            $form->disableViewButton(); // 去掉跳转详情页按钮
        });
    }
}
